import { d as defineEventHandler, g as getRouterParam, c as createError, q as query, r as readFormData, w as wrapZodError, a as getRequestIP, u as useRuntimeConfig, s as sendEmail, b as setResponseStatus } from '../../../nitro/nitro.mjs';
import ejs from 'ejs';
import logger from 'consola';
import { z } from 'zod/v4';
import 'nodemailer';
import 'node:http';
import 'node:https';
import 'node:events';
import 'node:buffer';
import 'node:fs';
import 'node:path';
import 'node:crypto';
import 'crypto';
import 'fs-extra';
import 'path';
import 'node:url';
import 'fs';
import 'mime-types';
import '@antfu/utils';
import 'dot-prop';
import 'redis';
import 'pg';
import 'sequelize';
import 'pg-connection-string';
import 'colorette';
import '@aws-sdk/client-s3';
import 'jsonwebtoken';
import 'nanoid';
import 'pluralize-esm';
import 'argon2';
import 'sharp';
import 'accept-language-parser';

z.config(z.locales.ru());
const formSchema = (formFields) => {
  const schema = {};
  for (const field of formFields) {
    switch (field.type) {
      case "checkbox":
        {
          let f = z.coerce.boolean();
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultBoolean);
        }
        break;
      case "date":
        {
          let f = z.coerce.date().min(field.min).max(field.max);
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultDate);
        }
        break;
      case "time":
        {
          let f = z.string().regex(/^([01]\d|2[0-3]):([0-5]\d)$/, {
            message: "\u0412\u0440\u0435\u043C\u044F \u0434\u043E\u043B\u0436\u043D\u043E \u0431\u044B\u0442\u044C \u0432 \u0444\u043E\u0440\u043C\u0430\u0442\u0435 \u0427\u0427:\u041C\u041C"
          });
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(
            field.defaultTime ? new Date(field.defaultTime).toISOString().split("T")[1].slice(0, 5) : void 0
          );
        }
        break;
      case "email":
        {
          let f = z.email();
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultString);
        }
        break;
      case "url":
        {
          let f = z.url();
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultString);
        }
        break;
      case "number":
      case "range":
        {
          let f = z.coerce.number().min(field.min).max(field.max);
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultNumber);
        }
        break;
      case "select":
      case "radio":
        {
          let f = z.enum(field.items.map((item) => item.value));
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultString);
        }
        break;
      case "password":
      case "tel":
      case "text":
      case "textarea":
        {
          let f = z.string();
          if (field.minLength > 0) {
            f = f.min(field.minLength);
          }
          if (field.maxLength > 0) {
            f = f.max(field.maxLength);
          }
          if (!field.required) {
            f = f.nullable();
          }
          schema[field.name] = f.default(field.defaultString);
        }
        break;
    }
  }
  return z.object(schema);
};
const _id__post = defineEventHandler(async (event) => {
  const id = z.coerce.number().int().safeParse(getRouterParam(event, "id"));
  if (!id.success) {
    throw createError({ status: 404 });
  }
  const form = await query("forms").select({ id: true, title: true, description: true, fields: true }).where("id", "=", id.data).first();
  if (!form) {
    throw createError({ status: 404 });
  }
  const schema = formSchema(form.fields);
  const rawFormData = await readFormData(event).catch(() => {
    throw createError({
      statusCode: 400,
      message: 'Content-Type was not one of "multipart/form-data" or "application/x-www-form-urlencoded".'
    });
  });
  const formData = form.fields.reduce(
    (acc, field) => {
      acc[field.name] = rawFormData.get(field.name);
      return acc;
    },
    {}
  );
  const sanitizedForm = schema.safeParse(formData);
  if (!sanitizedForm.success) {
    throw wrapZodError(sanitizedForm.error);
  }
  const clientIp = getRequestIP(event, {
    xForwardedFor: useRuntimeConfig().isBehindProxy
  });
  await query("form-submissions").create({
    form: id.data,
    ip: clientIp,
    json: JSON.stringify(sanitizedForm.data, null, 2)
  }).then((res) => {
    logger.info("New form submission received");
  }).catch((e) => {
    logger.error("Failed to save form submission in DB", e);
    throw e;
  });
  const notificationTask = query("server").read().then(async (cfg) => {
    if (!cfg.formNotificationsEnabled) {
      return;
    }
    if (!cfg.smtpEnabled) {
      logger.error(
        "Failed to send form submission notification because of SMTP is not configured"
      );
      return;
    }
    await sendEmail({
      to: cfg.formNotificationsReceivers.map(({ email }) => email).join(", "),
      from: cfg.smtpSender,
      html: ejs.render(cfg.formNotificationsTemplate, {
        form: {
          id: form.id,
          title: form.title,
          description: form.description
        },
        ip: clientIp,
        fields: Object.entries(sanitizedForm.data).map(([name, value]) => {
          var _a;
          return {
            name: ((_a = form.fields.find((x) => x.name === name)) == null ? void 0 : _a.label) || name,
            value
          };
        })
      })
    }).then(() => {
      logger.success("Form submission notification sent");
    }).catch((e) => {
      logger.error("Failed to send form submission notification", e);
    });
  });
  event.waitUntil(notificationTask);
  setResponseStatus(event, 201);
});

export { _id__post as default };
//# sourceMappingURL=_id_.post.mjs.map
